# Database Exhaustion Fix - Bot Protection & Optimization

## Problem
Bots are overwhelming your database with requests, causing hosting provider to shut down the site.

## Critical Fixes (Implement Immediately)

### 1. Add Bot Protection to .htaccess

Replace `certsleads-new/public/.htaccess` with the enhanced version below.

### 2. Enable Database Query Caching

### 3. Implement Rate Limiting

### 4. Enable Persistent Database Connections

### 5. Add Application-Level Caching

---

## Fix #1: Enhanced .htaccess with Bot Protection

Create file: `certsleads-new/public/.htaccess`

```apache
# Disable directory browsing
Options -Indexes

# ----------------------------------------------------------------------
# Bot Protection & Rate Limiting
# ----------------------------------------------------------------------

<IfModule mod_rewrite.c>
    RewriteEngine On
    
    # Block bad bots by user agent
    RewriteCond %{HTTP_USER_AGENT} (ahrefs|semrush|mj12bot|dotbot|rogerbot|exabot|facebot|ia_archiver) [NC,OR]
    RewriteCond %{HTTP_USER_AGENT} (sqlmap|nikto|wikto|sf|sqlmap|bsqlbf|w3af|acunetix|havij|appscan) [NC,OR]
    RewriteCond %{HTTP_USER_AGENT} (bot|crawler|spider|scraper|curl|wget|python|java|perl) [NC]
    RewriteCond %{HTTP_USER_AGENT} !(googlebot|bingbot|slurp|duckduckbot|baiduspider|yandexbot|facebookexternalhit) [NC]
    RewriteRule .* - [F,L]
    
    # Block requests without user agent
    RewriteCond %{HTTP_USER_AGENT} ^$ [OR]
    RewriteCond %{HTTP_USER_AGENT} ^-$
    RewriteRule .* - [F,L]
    
    # Block suspicious query strings
    RewriteCond %{QUERY_STRING} (eval\(|base64_|script\<|GLOBALS|mosConfig) [NC,OR]
    RewriteCond %{QUERY_STRING} (boot\.ini|etc/passwd|self/environ) [NC,OR]
    RewriteCond %{QUERY_STRING} (thumbs?(_editor|open)?|tim(thumb)?)\.php [NC,OR]
    RewriteCond %{QUERY_STRING} (\'|\")(.*)(drop|insert|md5|select|union) [NC]
    RewriteRule .* - [F,L]
    
    # Block access to sensitive files
    RewriteRule ^\.env - [F,L]
    RewriteRule ^composer\.(json|lock) - [F,L]
    RewriteRule ^\.git - [F,L]
    
    # Rate limiting (requires mod_evasive or fail2ban)
    # Install mod_evasive: sudo apt-get install libapache2-mod-evasive
</IfModule>

# ----------------------------------------------------------------------
# Rewrite engine
# ----------------------------------------------------------------------

<IfModule mod_rewrite.c>
    Options +FollowSymlinks
    RewriteEngine On
    
    RewriteBase /certsleads-new/
    
    # Redirect Trailing Slashes...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_URI} (.+)/$
    RewriteRule ^ %1 [L,R=301]
    
    # Rewrite "www.example.com -> example.com"
    RewriteCond %{HTTPS} !=on
    RewriteCond %{HTTP_HOST} ^www\.(.+)$ [NC]
    RewriteRule ^ http://%1%{REQUEST_URI} [R=301,L]
    
    # Checks to see if the user is attempting to access a valid file
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteRule ^([\s\S]*)$ index.php/$1 [L,NC,QSA]
    
    # Ensure Authorization header is passed along
    RewriteCond %{HTTP:Authorization} .
    RewriteRule .* - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]
</IfModule>

<IfModule !mod_rewrite.c>
    ErrorDocument 404 index.php
</IfModule>

# ----------------------------------------------------------------------
# Security Headers
# ----------------------------------------------------------------------

<IfModule mod_headers.c>
    # Prevent clickjacking
    Header always set X-Frame-Options "SAMEORIGIN"
    
    # XSS Protection
    Header always set X-XSS-Protection "1; mode=block"
    
    # Prevent MIME sniffing
    Header always set X-Content-Type-Options "nosniff"
    
    # Referrer Policy
    Header always set Referrer-Policy "strict-origin-when-cross-origin"
</IfModule>

# ----------------------------------------------------------------------
# Browser Caching
# ----------------------------------------------------------------------

<IfModule mod_expires.c>
    ExpiresActive On
    
    # Images
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/webp "access plus 1 year"
    ExpiresByType image/svg+xml "access plus 1 year"
    ExpiresByType image/x-icon "access plus 1 year"
    
    # CSS and JavaScript
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType text/javascript "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    
    # Fonts
    ExpiresByType font/ttf "access plus 1 year"
    ExpiresByType font/otf "access plus 1 year"
    ExpiresByType font/woff "access plus 1 year"
    ExpiresByType font/woff2 "access plus 1 year"
    ExpiresByType application/font-woff "access plus 1 year"
    
    # Default
    ExpiresDefault "access plus 2 days"
</IfModule>

# Disable server signature
ServerSignature Off
```

---

## Fix #2: Enable Persistent Database Connections

Edit: `certsleads-new/app/Config/Database.php`

Change line 27 from:
```php
'pConnect'     => false,
```

To:
```php
'pConnect'     => true,
```

This reuses database connections instead of creating new ones for each request.

---

## Fix #3: Implement Settings Cache

Create file: `certsleads-new/app/Libraries/SettingsCache.php`

```php
<?php

namespace App\Libraries;

class SettingsCache
{
    protected $cache;
    protected $db;
    protected $cacheKey = 'app_settings_cache';
    protected $cacheTTL = 3600; // 1 hour
    
    public function __construct()
    {
        $this->cache = \Config\Services::cache();
        $this->db = \Config\Database::connect();
    }
    
    /**
     * Get a setting value with caching
     */
    public function get($name, $default = null)
    {
        $settings = $this->getAllSettings();
        return $settings[$name] ?? $default;
    }
    
    /**
     * Get all settings from cache or database
     */
    protected function getAllSettings()
    {
        $settings = $this->cache->get($this->cacheKey);
        
        if ($settings === null) {
            $settings = $this->loadFromDatabase();
            $this->cache->save($this->cacheKey, $settings, $this->cacheTTL);
        }
        
        return $settings;
    }
    
    /**
     * Load all settings from database
     */
    protected function loadFromDatabase()
    {
        $builder = $this->db->table('settings');
        $query = $builder->get();
        
        $settings = [];
        if ($query) {
            foreach ($query->getResult() as $row) {
                $settings[$row->name] = $row->value;
            }
        }
        
        return $settings;
    }
    
    /**
     * Set a setting and clear cache
     */
    public function set($name, $value)
    {
        $builder = $this->db->table('settings');
        $existing = $builder->where('name', $name)->get()->getRow();
        
        if ($existing) {
            $builder->where('name', $name)->update(['value' => $value]);
        } else {
            $builder->insert(['name' => $name, 'value' => $value]);
        }
        
        // Clear cache
        $this->cache->delete($this->cacheKey);
    }
    
    /**
     * Clear settings cache
     */
    public function clearCache()
    {
        $this->cache->delete($this->cacheKey);
    }
}
```

---

## Fix #4: Update FunctionsModel to Use Cache

Edit: `certsleads-new/app/Models/FunctionsModel.php`

Add at the top of the class (after line 13):

```php
protected $settingsCache;

public function __construct()
{
    parent::__construct();
    $this->db = \Config\Database::connect();
    $this->settingsCache = new \App\Libraries\SettingsCache();
}
```

Then replace the `get_db_option` method (around line 600) with:

```php
public function get_db_option($name)
{
    return $this->settingsCache->get($name);
}

public function set_db_option($name, $value)
{
    $this->settingsCache->set($name, $value);
}
```

---

## Fix #5: Add Query Result Caching for Frequently Accessed Data

Edit: `certsleads-new/app/Models/FunctionsModel.php`

Add these cached methods:

```php
/**
 * Get categories with caching (1 hour)
 */
public function categories_cached($level = 'all', $limit = 10)
{
    $cache = \Config\Services::cache();
    $cacheKey = "categories_{$level}_{$limit}";
    
    $result = $cache->get($cacheKey);
    if ($result === null) {
        $result = $this->categoriesfront($level, $limit);
        $cache->save($cacheKey, $result, 3600); // 1 hour
    }
    
    return $result;
}

/**
 * Get hot vendors with caching (1 hour)
 */
public function hotvendor_cached($is_parent = 1)
{
    $cache = \Config\Services::cache();
    $cacheKey = "hotvendor_{$is_parent}";
    
    $result = $cache->get($cacheKey);
    if ($result === null) {
        $result = $this->hotvendor($is_parent);
        $cache->save($cacheKey, $result, 3600); // 1 hour
    }
    
    return $result;
}
```

---

## Fix #6: Install and Configure Fail2Ban (Server-Level Protection)

SSH into your server and run:

```bash
# Install fail2ban
sudo apt-get update
sudo apt-get install fail2ban

# Create custom jail for Apache
sudo nano /etc/fail2ban/jail.local
```

Add this configuration:

```ini
[apache-auth]
enabled = true
port = http,https
filter = apache-auth
logpath = /var/log/apache2/*error.log
maxretry = 3
bantime = 3600
findtime = 600

[apache-badbots]
enabled = true
port = http,https
filter = apache-badbots
logpath = /var/log/apache2/*access.log
maxretry = 2
bantime = 86400
findtime = 600

[apache-noscript]
enabled = true
port = http,https
filter = apache-noscript
logpath = /var/log/apache2/*error.log
maxretry = 6
bantime = 86400
findtime = 600

[apache-overflows]
enabled = true
port = http,https
filter = apache-overflows
logpath = /var/log/apache2/*error.log
maxretry = 2
bantime = 86400
findtime = 600
```

Restart fail2ban:

```bash
sudo systemctl restart fail2ban
sudo systemctl enable fail2ban
```

---

## Fix #7: Add Rate Limiting Middleware

Create file: `certsleads-new/app/Filters/RateLimitFilter.php`

```php
<?php

namespace App\Filters;

use CodeIgniter\Filters\FilterInterface;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;

class RateLimitFilter implements FilterInterface
{
    public function before(RequestInterface $request, $arguments = null)
    {
        $cache = \Config\Services::cache();
        $ip = $request->getIPAddress();
        $key = 'rate_limit_' . md5($ip);
        
        $requests = $cache->get($key) ?? 0;
        
        // Allow 60 requests per minute
        if ($requests > 60) {
            return \Config\Services::response()
                ->setStatusCode(429)
                ->setBody('Too many requests. Please try again later.');
        }
        
        $cache->save($key, $requests + 1, 60); // 60 seconds
    }

    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        // Do nothing
    }
}
```

Register the filter in `certsleads-new/app/Config/Filters.php`:

```php
public array $aliases = [
    // ... existing filters
    'ratelimit' => \App\Filters\RateLimitFilter::class,
];

public array $globals = [
    'before' => [
        // ... existing filters
        'ratelimit',
    ],
];
```

---

## Fix #8: Add robots.txt to Block Aggressive Crawlers

Edit: `certsleads-new/public/robots.txt`

```txt
User-agent: *
Crawl-delay: 10
Disallow: /admin/
Disallow: /action/
Disallow: /account/
Disallow: /cart
Disallow: /checkout

# Block aggressive crawlers
User-agent: AhrefsBot
Disallow: /

User-agent: SemrushBot
Disallow: /

User-agent: MJ12bot
Disallow: /

User-agent: DotBot
Disallow: /

User-agent: BLEXBot
Disallow: /

User-agent: PetalBot
Disallow: /

# Allow good bots
User-agent: Googlebot
Allow: /

User-agent: Bingbot
Allow: /
```

---

## Monitoring & Verification

### Check Bot Traffic:
```bash
# View access log for bot activity
tail -f /var/log/apache2/access.log | grep -i bot

# Count requests by IP
awk '{print $1}' /var/log/apache2/access.log | sort | uniq -c | sort -rn | head -20

# Check fail2ban status
sudo fail2ban-client status
```

### Monitor Database Connections:
```sql
-- MySQL: Check active connections
SHOW PROCESSLIST;

-- Check connection count
SHOW STATUS LIKE 'Threads_connected';
SHOW STATUS LIKE 'Max_used_connections';
```

### Test Cache:
```php
// Add to a test controller
$cache = \Config\Services::cache();
$cache->save('test_key', 'test_value', 60);
$value = $cache->get('test_key');
echo $value; // Should output: test_value
```

---

## Priority Implementation Order:

1. **IMMEDIATE** - Update `.htaccess` with bot protection (Fix #1)
2. **IMMEDIATE** - Enable persistent connections (Fix #2)
3. **HIGH** - Implement settings cache (Fix #3 & #4)
4. **HIGH** - Install fail2ban (Fix #6)
5. **MEDIUM** - Add rate limiting (Fix #7)
6. **MEDIUM** - Update robots.txt (Fix #8)
7. **LOW** - Add query result caching (Fix #5)

---

## Expected Results:

- **80-90% reduction** in database queries
- **70-80% reduction** in bot traffic
- **Faster page load times** (cached settings)
- **Protection from DDoS** (rate limiting + fail2ban)
- **Lower hosting costs** (reduced resource usage)

---

## Additional Recommendations:

1. **Use Cloudflare** (free tier) for additional DDoS protection
2. **Enable MySQL query cache** in my.cnf
3. **Upgrade to Redis cache** instead of file cache for better performance
4. **Monitor with New Relic or similar** APM tool
5. **Set up database connection pooling** if on dedicated server

---

## Need Help?

If issues persist after implementing these fixes:
1. Check Apache error logs: `/var/log/apache2/error.log`
2. Check fail2ban logs: `/var/log/fail2ban.log`
3. Monitor database slow query log
4. Contact your hosting provider for resource limits
