<?php

namespace App\Controllers;

class Action extends BaseController
{
    /**
     * Add exam product to cart. Expects POST: product, sku.
     */
    public function add_to_cart()
    {
        $product = $this->request->getPost('product');
        $sku = $this->request->getPost('sku');

        if (! $product || ! $sku) {
            return redirect()->to(base_url('cart'));
        }

        $check1 = $this->Functions->exam('sku', $sku);
        if (! $check1) {
            return redirect()->to(base_url('cart'));
        }

        $exam = $this->Functions->exam('id', $check1->id);
        if (! $exam || ! $exam->sale_price || ! $exam->title) {
            return redirect()->to(base_url('cart'));
        }

        $qty   = 1;
        $price = (float) $exam->sale_price;
        $item  = [
            'id'       => $sku,
            'rowid'    => $sku,
            'sku'      => $sku,
            'qty'      => $qty,
            'price'    => $price,
            'subtotal' => $price * $qty,
            'name'     => $exam->title,
            'p_type'   => 'exam',
            'options'  => ['product' => $product],
        ];

        $cart = $this->session->get('cart') ?? [];
        if (! is_array($cart)) {
            $cart = [];
        }
        $cart[] = $item;
        $this->session->set('cart', $cart);

        return redirect()->to(base_url('cart'));
    }

    /**
     * Clear cart.
     */
    public function destroy_cart()
    {
        $this->session->remove('cart');
        return redirect()->to(base_url('cart'));
    }

    /**
     * Remove one item from cart by rowid (sku).
     */
    public function remove_from_cart($rowid)
    {
        $cart = $this->session->get('cart') ?? [];
        if (! is_array($cart)) {
            return redirect()->to(base_url('cart'));
        }
        $cart = array_values(array_filter($cart, static function ($item) use ($rowid) {
            $id = $item['rowid'] ?? $item['id'] ?? $item['sku'] ?? null;
            return $id !== $rowid;
        }));
        $this->session->set('cart', $cart);
        return redirect()->to(base_url('cart'));
    }

    /**
     * Update cart: apply coupon if submitted, optionally update item qtys.
     */
    public function update_cart()
    {
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('cart'));
        }

        $post = $this->request->getPost();
        $cart = $this->session->get('cart') ?? [];
        if (! is_array($cart)) {
            $cart = [];
        }

        // Check if coupon is in POST data directly
        $couponCode = strtoupper(trim((string) $this->request->getPost('coupon')));
        if ($couponCode !== '') {
            $coupon = $this->Functions->coupon('code', $couponCode);
            if ($coupon && isset($coupon->id)) {
                $this->session->set('coupon', $coupon->id);
                $this->session->remove('promo_auto_attempted');
                $this->session->setFlashdata('msg', 'Promotional Code "' . $couponCode . '" applied successfully.');
                $this->session->setFlashdata('msgClass', 'alert-success');
            } else {
                $this->session->setFlashdata('msg', 'Invalid promotional code: "' . $couponCode . '"');
                $this->session->setFlashdata('msgClass', 'alert-danger');
            }
        }

        foreach ($post as $name => $item) {
            if (is_array($item) && isset($item['rowid']) && isset($item['qty'])) {
                $rowid = $item['rowid'];
                $qty   = max(1, (int) $item['qty']);
                foreach ($cart as $i => $c) {
                    $id = $c['rowid'] ?? $c['id'] ?? $c['sku'] ?? null;
                    if ($id === $rowid) {
                        $cart[$i]['qty']      = $qty;
                        $cart[$i]['subtotal'] = ($c['price'] ?? 0) * $qty;
                        break;
                    }
                }
            }
        }

        $this->session->set('cart', $cart);

        return redirect()->to(base_url('cart'));
    }

    /**
     * Remove applied coupon.
     */
    public function remove_coupon()
    {
        $this->session->remove('coupon');
        $this->session->remove('promo_auto_attempted');
        $this->session->setFlashdata('msg', 'Promotional Code Removed Successfully');
        $this->session->setFlashdata('msgClass', 'alert-success');
        return redirect()->to(base_url('cart'));
    }

    /**
     * Register with reCAPTCHA (signin page form). POST: name, email, password, confirmPassword, recaptcha_response.
     */
    public function registers()
    {
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('account/signin'));
        }
        $recaptchaResponse = $this->request->getPost('recaptcha_response');
        $secret = $this->Functions->get_db_option('recaptcha_secret') ?: '6LeEYPArAAAAALeqE0WAPSB41xlULM9tyB76tNTk';
        $validRecaptcha = false;
        if ($recaptchaResponse) {
            $url = 'https://www.google.com/recaptcha/api/siteverify';
            $context = stream_context_create([
                'http' => [
                    'header'  => "Content-type: application/x-www-form-urlencoded\r\n",
                    'method'  => 'POST',
                    'content' => http_build_query([
                        'secret'   => $secret,
                        'response' => $recaptchaResponse,
                        'remoteip' => $this->request->getIPAddress(),
                    ]),
                ],
            ]);
            $resp = @file_get_contents($url, false, $context);
            $result = $resp ? json_decode($resp, true) : null;
            $validRecaptcha = $result && ! empty($result['success']) && (float) ($result['score'] ?? 0) > 0.5;
        }
        $regNext = $this->request->getGet('next') ?? base_url('account');
        $regUrl  = base_url('account/signin') . '?next=' . urlencode($regNext);
        if (! $validRecaptcha) {
            return redirect()->to($regUrl . '&register_error=recaptcha');
        }
        $email = trim((string) $this->request->getPost('email'));
        $name  = trim((string) $this->request->getPost('name'));
        $password = $this->request->getPost('password');
        $confirmPassword = $this->request->getPost('confirmPassword');
        if (! $email || ! filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return redirect()->to($regUrl . '&register_error=email');
        }
        if (! $name) {
            return redirect()->to($regUrl . '&register_error=name');
        }
        if (! $password || strlen($password) < 4) {
            return redirect()->to($regUrl . '&register_error=password');
        }
        if ($password !== $confirmPassword) {
            return redirect()->to($regUrl . '&register_error=match');
        }
        $u = $this->Functions->user('email', $email);
        if ($u) {
            return redirect()->to($regUrl . '&register_error=exists');
        }
        $this->Functions->register($email, $name, $password);
        $u = $this->Functions->user('email', $email);
        if (! $u) {
            return redirect()->to($regUrl . '&register_error=failed');
        }
        $this->session->set('user_id', $u->id);
        return redirect()->to($regNext);
    }

    /**
     * Simple register (register page form). POST: name, email, password, confirmPassword.
     */
    public function register()
    {
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('account/register'));
        }
        $email = trim((string) $this->request->getPost('email'));
        $name  = trim((string) $this->request->getPost('name'));
        $password = $this->request->getPost('password');
        $confirmPassword = $this->request->getPost('confirmPassword');
        $next    = $this->request->getGet('next') ?? base_url('account');
        $regUrl  = base_url('account/register') . '?next=' . urlencode($next);
        if (! $email || ! filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return redirect()->to($regUrl . '&register_error=email');
        }
        if (! $name) {
            return redirect()->to($regUrl . '&register_error=name');
        }
        if (! $password || strlen($password) < 4) {
            return redirect()->to($regUrl . '&register_error=password');
        }
        if ($password !== $confirmPassword) {
            return redirect()->to($regUrl . '&register_error=match');
        }
        $u = $this->Functions->user('email', $email);
        if ($u) {
            return redirect()->to($regUrl . '&register_error=exists');
        }
        $this->Functions->register($email, $name, $password);
        $u = $this->Functions->user('email', $email);
        if (! $u) {
            return redirect()->to($regUrl . '&register_error=failed');
        }
        $this->session->set('user_id', $u->id);
        return redirect()->to($next);
    }

    /**
     * Session create (sign-in) or destroy (logout). GET/POST action/session/create, GET action/session/destroy.
     */
    public function session($action = 'destroy')
    {
        $next = $this->request->getGet('next') ?? base_url();
        if ($action === 'create') {
            if ($this->request->getMethod() !== 'POST') {
                return redirect()->to(base_url('account/signin') . '?next=' . urlencode($next));
            }
            
            $email = trim((string) $this->request->getPost('email'));
            $password = $this->request->getPost('password');
            $signinUrl = base_url('account/signin') . '?next=' . urlencode($next);
            if (! $email || ! filter_var($email, FILTER_VALIDATE_EMAIL)) {
                return redirect()->to($signinUrl . '&login_error=email');
            }
            if (! $password) {
                return redirect()->to($signinUrl . '&login_error=password');
            }
            if ($this->Functions->check_user($email, $password)) {
                $u = $this->Functions->user('email', $email);
                $this->session->set('user_id', $u->id);
                return redirect()->to($next);
            }
            return redirect()->to($signinUrl . '&login_error=invalid');
        }
        $this->session->remove('user_id');
        return redirect()->to($next);
    }

    /**
     * Admin login. POST from admin login form (email, password).
     * Validates credentials, ensures user is admin (is_admin == 1), then sets session and redirects to admin.
     */
    public function admin_login()
    {
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin_login'));
        }

        $email    = trim((string) $this->request->getPost('email'));
        $password = $this->request->getPost('password');

        if (! $email || ! filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->session->setFlashdata('msg', 'Please enter a valid email.');
            return redirect()->to(base_url('admin_login'));
        }
        if (! $password) {
            $this->session->setFlashdata('msg', 'Please enter your password.');
            return redirect()->to(base_url('admin_login'));
        }
        if (! $this->Functions->check_user($email, $password)) {
            $this->session->setFlashdata('msg', 'Invalid email or password.');
            return redirect()->to(base_url('admin_login'));
        }

        $u = $this->Functions->user('email', $email);
        if (! $u || (string) $u->is_admin !== '1') {
            $this->session->setFlashdata('msg', 'Access denied. Administrator only.');
            return redirect()->to(base_url('admin_login'));
        }

        $this->session->set('user_id', $u->id);
        return redirect()->to(base_url('admin'));
    }

    /**
     * Forget password. GET: show form. POST: email -> send reset link.
     */
    public function forget()
    {
        if ($this->request->getMethod() === 'post') {
            $email = trim((string) $this->request->getPost('email'));
            if (! $email || ! filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $this->session->setFlashdata('loginError', 'Please enter a valid email.');
                return redirect()->to(base_url('action/forget'));
            }
            $u = $this->Functions->check_user_email($email);
            if ($u) {
                $token = $this->Functions->userTokenUpdate($u->email, $u->id);
                $link = base_url('action/resetpass/' . $token);
                $body = "Dear {$u->name},\n\nPlease click the link below to reset your password:\n{$link}\n\nIf you have questions, contact us at support@certsleads.com.\n\nRegards,\ncertsleads.com";
                $emailService = \Config\Services::email();
                $emailService->setFrom($this->Functions->get_db_option('mail_from') ?: 'info@certsleads.com', 'certsleads.com');
                $emailService->setTo($u->email);
                $emailService->setSubject('Reset your password at certsleads.com');
                $emailService->setMessage($body);
                $emailService->send();
                $this->session->setFlashdata('mailsuccess', 'An email has been sent to your registered address. Please follow the instructions.');
            } else {
                $this->session->setFlashdata('loginError', 'Invalid email.');
            }
            return redirect()->to(base_url('action/forget'));
        }
        return $this->template('forget', ['title' => 'Forget Password', 'class' => 'signin frontend account']);
    }

    /**
     * Reset password. GET with token: show form. POST: token, password, cPassword.
     */
    public function resetpass($token = '')
    {
        if ($this->request->getMethod() === 'post') {
            $token = trim((string) $this->request->getPost('token'));
            $password = $this->request->getPost('password');
            $cPassword = $this->request->getPost('cPassword');
            if (! $token) {
                $this->session->setFlashdata('tokenError', 'Invalid token. Please request a new reset link.');
                return redirect()->to(base_url('action/resetpass'));
            }
            if (! $password || strlen($password) < 4) {
                $this->session->setFlashdata('tokenError', 'Please enter a new password (min 4 characters).');
                return redirect()->to(base_url('action/resetpass/' . $token));
            }
            if ($password !== $cPassword) {
                $this->session->setFlashdata('tokenError', 'Passwords do not match.');
                return redirect()->to(base_url('action/resetpass/' . $token));
            }
            $u = $this->Functions->check_user_token($token);
            if ($u) {
                $this->Functions->userResetPassUpdate($password, $u->id);
                $this->session->setFlashdata('resetsuccess', 'Your password has been reset. Please sign in.');
                return redirect()->to(base_url('action/resetpass'));
            }
            $this->session->setFlashdata('tokenError', 'Invalid token. Please request a new reset link.');
            return redirect()->to(base_url('action/resetpass/' . $token));
        }
        return $this->template('restpassword', ['title' => 'Reset Password', 'token' => $token, 'class' => 'signin frontend account']);
    }

    /**
     * Update profile. POST: name, email, gender, address, city, state, country, zip, vat, phone, mobile.
     */
    public function update_profile()
    {
        if (! $this->session->get('user_id')) {
            return redirect()->to(base_url('account/signin'));
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('account/settings'));
        }
        $name = trim((string) $this->request->getPost('name'));
        $email = trim((string) $this->request->getPost('email'));
        $gender = trim((string) $this->request->getPost('gender'));
        $address = trim((string) $this->request->getPost('address'));
        $city = trim((string) $this->request->getPost('city'));
        $state = trim((string) $this->request->getPost('state'));
        $country = trim((string) $this->request->getPost('country'));
        $zip = trim((string) $this->request->getPost('zip'));
        $vat = trim((string) $this->request->getPost('vat'));
        $phone = trim((string) $this->request->getPost('phone'));
        $mobile = trim((string) $this->request->getPost('mobile'));
        if (! $email || ! filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->session->setFlashdata('msg', 'Please enter a valid email.');
            $this->session->setFlashdata('msgClass', 'alert-danger');
            return redirect()->to(base_url('account/settings'));
        }
        if (! $name) {
            $this->session->setFlashdata('msg', 'Please enter your name.');
            $this->session->setFlashdata('msgClass', 'alert-danger');
            return redirect()->to(base_url('account/settings'));
        }
        $id = $this->session->get('user_id');
        $user = $this->Functions->user('id', $id);
        if (! $user) {
            return redirect()->to(base_url('account/signin'));
        }
        if ($user->email !== $email) {
            $existing = $this->Functions->user('email', $email);
            if ($existing) {
                $this->session->setFlashdata('msg', 'A user with this email already exists.');
                $this->session->setFlashdata('msgClass', 'alert-danger');
                return redirect()->to(base_url('account/settings'));
            }
        }
        $this->Functions->update_profile($id, $name, $email, $gender, $address, $city, $state, $country, $zip, $vat, $phone, $mobile);
        $this->session->setFlashdata('msg', 'Profile updated successfully.');
        $this->session->setFlashdata('msgClass', 'alert-success');
        return redirect()->to(base_url('account/settings'));
    }

    /**
     * Change password. POST: oPassword, password, cPassword.
     */
    public function changepass()
    {
        if (! $this->session->get('user_id')) {
            return redirect()->to(base_url('account/signin'));
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('account/changepass'));
        }
        $oPassword = $this->request->getPost('oPassword');
        $password = $this->request->getPost('password');
        $cPassword = $this->request->getPost('cPassword');
        if (! $oPassword || ! $password || ! $cPassword) {
            $this->session->setFlashdata('msg', 'All fields are required.');
            $this->session->setFlashdata('msgClass', 'alert-danger');
            return redirect()->to(base_url('account/changepass'));
        }
        if ($password !== $cPassword) {
            $this->session->setFlashdata('msg', 'New passwords do not match.');
            $this->session->setFlashdata('msgClass', 'alert-danger');
            return redirect()->to(base_url('account/changepass'));
        }
        $u = $this->Functions->user('id', $this->session->get('user_id'));
        if (! $u || ! $this->Functions->check_user($u->email, $oPassword)) {
            $this->session->setFlashdata('msg', 'Incorrect current password.');
            $this->session->setFlashdata('msgClass', 'alert-danger');
            return redirect()->to(base_url('account/changepass'));
        }
        $this->Functions->changepass($password, $u->id);
        $this->session->setFlashdata('msg', 'Password updated.');
        $this->session->setFlashdata('msgClass', 'alert-success');
        return redirect()->to(base_url('account/changepass'));
    }

    /**
     * Standard checkout (Ziina). POST: email. Creates/fetches user, creates payment intent, redirects to Ziina.
     */
    public function usercheckout()
    {
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('cart'));
        }
        $email = trim((string) $this->request->getPost('email'));
        if ($email === '') {
            return redirect()->to(base_url('cart'));
        }
        helper('cart');
        $cart = $this->session->get('cart') ?? [];
        if (! is_array($cart)) {
            $cart = [];
        }
        $grandTotal = cart_total($cart);
        $couponId  = $this->session->get('coupon');
        $charged   = $grandTotal;
        if ($couponId) {
            $coupon = $this->Functions->coupon('id', $couponId);
            if ($coupon && isset($coupon->discount)) {
                $charged = $grandTotal - (($coupon->discount / 100) * $grandTotal);
            }
        }
        $sku = $this->Functions->get_sku_cart();
        $u   = $this->Functions->user('email', $email);
        if ($u) {
            $user = $u->id;
            $this->session->set('user_id', $u->id);
        } else {
            $name     = explode('@', $email)[0] ?? 'User';
            $password = bin2hex(random_bytes(8));
            $this->Functions->register($email, $name, $password);
            $new = $this->Functions->user('email', $email);
            if (! $new) {
                $this->session->setFlashdata('msg', 'Could not create account. Please try again.');
                $this->session->setFlashdata('msgClass', 'alert-danger');
                return redirect()->to(base_url('cart'));
            }
            $user = $new->id;
            $this->session->set('user_id', $new->id);
        }
        if (! $charged || ! $sku || ! $user) {
            $this->session->setFlashdata('msg', 'Order processing failed. Please contact support!');
            $this->session->setFlashdata('msgClass', 'alert-danger');
            return redirect()->to(base_url('cart'));
        }
        $productNames = [];
        foreach ($cart as $item) {
            $productNames[] = $item['name'] ?? '';
        }
        $productNamesStr = implode(', ', array_filter($productNames));
        $ziinaToken      = $this->Functions->get_db_option('ziina_api_token') ?: 'UWxXSipFHDbMj2NhcnNSmLvy0JxF9+SZARk1en/35496s8Gq7lGY1uTjEsj5PPt6';
        $ch              = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL            => 'https://api-v2.ziina.com/api/payment_intent',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => json_encode([
                'allow_tips'   => null,
                'amount'       => (int) round($charged * 100),
                'currency_code' => 'USD',
                'message'      => 'Purchased Exams: ' . $productNamesStr,
                'success_url'  => base_url() . 'action/success',
                'cancel_url'   => base_url() . 'action/cancel',
                'failure_url'  => base_url() . 'action/failed',
                'test'         => null,
            ]),
            CURLOPT_HTTPHEADER     => [
                'Authorization: Bearer ' . $ziinaToken,
                'Content-Type: application/json',
            ],
        ]);
        $response = curl_exec($ch);
        $err      = curl_error($ch);
        curl_close($ch);
        if ($err) {
            $this->session->setFlashdata('msg', 'Payment service error. Please try again.');
            $this->session->setFlashdata('msgClass', 'alert-danger');
            return redirect()->to(base_url('cart'));
        }
        $data = json_decode($response, true);
        if (! $data || empty($data['id']) || empty($data['redirect_url'])) {
            $this->session->setFlashdata('msg', 'Could not create payment. Please try again.');
            $this->session->setFlashdata('msgClass', 'alert-danger');
            return redirect()->to(base_url('cart'));
        }
        $this->Functions->update_user_payment_id($user, $data['id']);
        return redirect()->to($data['redirect_url']);
    }

    /**
     * PayPal checkout. POST: email. Creates/fetches user, redirects to PayPal or cart with message.
     */
    public function paypal_standard_checkout()
    {
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('cart'));
        }
        $email = trim((string) $this->request->getPost('email'));
        if ($email === '') {
            return redirect()->to(base_url('cart'));
        }
        helper('cart');
        $cart       = $this->session->get('cart') ?? [];
        $cart       = is_array($cart) ? $cart : [];
        $grandTotal = cart_total($cart);
        $couponId   = $this->session->get('coupon');
        $charged    = $grandTotal;
        if ($couponId) {
            $coupon = $this->Functions->coupon('id', $couponId);
            if ($coupon && isset($coupon->discount)) {
                $charged = $grandTotal - (($coupon->discount / 100) * $grandTotal);
            }
        }
        $sku = $this->Functions->get_sku_cart();
        $u   = $this->Functions->user('email', $email);
        if ($u) {
            $user = $u->id;
            $this->session->set('user_id', $u->id);
        } else {
            $name     = explode('@', $email)[0] ?? 'User';
            $password = bin2hex(random_bytes(8));
            $this->Functions->register($email, $name, $password);
            $new = $this->Functions->user('email', $email);
            if (! $new) {
                $this->session->setFlashdata('msg', 'Could not create account. Please try again.');
                $this->session->setFlashdata('msgClass', 'alert-danger');
                return redirect()->to(base_url('cart'));
            }
            $user = $new->id;
            $this->session->set('user_id', $new->id);
        }
        if (! $charged || ! $sku || ! $user) {
            $this->session->setFlashdata('msg', 'Order processing failed. Please contact support!');
            $this->session->setFlashdata('msgClass', 'alert-danger');
            return redirect()->to(base_url('cart'));
        }
        $business = $this->Functions->get_db_option('paypal_business_email') ?: '';
        $sandbox  = $this->Functions->get_db_option('paypal_sandbox');
        $paypalUrl = ($sandbox === 'true' || $sandbox === '1')
            ? 'https://www.sandbox.paypal.com/cgi-bin/webscr'
            : 'https://www.paypal.com/cgi-bin/webscr';
        if ($business === '') {
            $this->session->setFlashdata('msg', 'PayPal is not configured. Please use another payment method.');
            $this->session->setFlashdata('msgClass', 'alert-warning');
            return redirect()->to(base_url('cart'));
        }
        $returnUrl = base_url() . 'paypal/success';
        $cancelUrl = base_url() . 'paypal/cancel';
        $notifyUrl = base_url() . 'paypal/ipn';
        $fields    = [
            'business'      => $business,
            'cmd'           => '_xclick',
            'item_name'     => 'Paper Purchase',
            'item_number'   => $sku,
            'amount'        => $charged,
            'currency_code' => 'USD',
            'custom'        => $user,
            'return'        => $returnUrl,
            'cancel_return' => $cancelUrl,
            'notify_url'    => $notifyUrl,
            'rm'            => '2',
            'quantity'      => '1',
        ];
        $html = '<!DOCTYPE html><html><head><title>Redirecting to PayPal…</title></head><body>';
        $html .= '<form id="ppform" method="post" action="' . htmlspecialchars($paypalUrl) . '">';
        foreach ($fields as $k => $v) {
            $html .= '<input type="hidden" name="' . htmlspecialchars($k) . '" value="' . htmlspecialchars((string) $v) . '" />';
        }
        $html .= '<p>Redirecting to PayPal…</p><input type="submit" value="Continue to PayPal" /></form>';
        $html .= '<script>document.getElementById("ppform").submit();</script></body></html>';
        return $this->response->setBody($html)->setHeader('Content-Type', 'text/html');
    }

    /**
     * Ziina success callback. Verifies payment, adds order, clears cart, redirects to thank-you.
     */
    public function success()
    {
        $userId = $this->session->get('user_id');
        if (! $userId) {
            return redirect()->to(base_url('cart'));
        }
        $u = $this->Functions->user('id', $userId);
        if (! $u || empty($u->pid)) {
            return redirect()->to(base_url('cart'));
        }
        $ziinaToken = $this->Functions->get_db_option('ziina_api_token') ?: 'UWxXSipFHDbMj2NhcnNSmLvy0JxF9+SZARk1en/35496s8Gq7lGY1uTjEsj5PPt6';
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL            => 'https://api-v2.ziina.com/api/payment_intent/' . $u->pid,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST  => 'GET',
            CURLOPT_HTTPHEADER     => ['Authorization: Bearer ' . $ziinaToken],
        ]);
        $response = curl_exec($ch);
        curl_close($ch);
        $data = json_decode($response, true);
        if (! $data || ($data['status'] ?? '') !== 'completed') {
            return redirect()->to(base_url('cart'));
        }
        helper('cart');
        $cart       = $this->session->get('cart') ?? [];
        $cart       = is_array($cart) ? $cart : [];
        $grandTotal = cart_total($cart);
        $couponId   = $this->session->get('coupon');
        $charged    = $grandTotal;
        if ($couponId) {
            $coupon = $this->Functions->coupon('id', $couponId);
            if ($coupon && isset($coupon->discount)) {
                $charged = $grandTotal - (($coupon->discount / 100) * $grandTotal);
            }
        }
        $sku          = $this->Functions->get_sku_cart();
        $couponObj    = $couponId ? $this->Functions->coupon('id', $couponId) : null;
        $couponString = ($couponObj !== null && isset($couponObj->code)) ? $couponObj->code : '';
        $this->Functions->add_order($charged, $sku, (string) $userId, $couponString, date('Y-m-d H:i:s'));
        $this->session->remove('cart');
        $this->session->remove('coupon');
        return redirect()->to(base_url('thank-you'));
    }

    /**
     * Ziina cancel callback.
     */
    public function cancel()
    {
        $this->session->setFlashdata('msg', 'Payment was cancelled.');
        $this->session->setFlashdata('msgClass', 'alert-info');
        return redirect()->to(base_url('cart'));
    }

    /**
     * Ziina failure callback.
     */
    public function failed()
    {
        $this->session->setFlashdata('msg', 'Payment failed. Please try again.');
        $this->session->setFlashdata('msgClass', 'alert-danger');
        return redirect()->to(base_url('cart'));
    }

    /**
     * Ensure current user is admin; redirect to admin login if not. Returns redirect or null.
     */
    protected function requireAdmin()
    {
        $userId = $this->session->get('user_id');
        if (! $userId) {
            return redirect()->to(base_url('admin_login'));
        }
        $user = $this->Functions->user('id', $userId);
        if (! $user || (string) $user->is_admin !== '1') {
            return redirect()->to(base_url('admin_login'));
        }
        return null;
    }

    /**
     * Edit exam. POST from admin edit_exam form. Requires admin.
     */
    public function edit_exam($id)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/exam/' . $id));
        }
        $exam = $this->Functions->exam('id', $id);
        if (! $exam) {
            $this->session->setFlashdata('msg', 'Exam not found.');
            $this->session->setFlashdata('msgColor', 'red');
            return redirect()->to(base_url('admin/exams'));
        }
        $p = $this->request->getPost();
        $data = [];
        // Map form fields to DB columns (exams table uses 'update' not 'updated')
        $allowed = ['title', 'name', 'certification', 'category', 'questions', 'sale_price', 'regular_price', 'slug', 'path_slug', 'demo_slug'];
        foreach ($allowed as $key) {
            if (array_key_exists($key, $p)) {
                $data[$key] = $p[$key];
            }
        }
        if (isset($p['updated']) && $p['updated'] !== '') {
            $data['update'] = $p['updated'];
        }
        $data['on_homepage'] = isset($p['on_homepage']) && $p['on_homepage'] ? '1' : '0';
        $data['hot_exam']   = isset($p['hot_exam']) && $p['hot_exam'] ? '1' : '0';
        $data['top_exam']   = isset($p['top_exam']) && $p['top_exam'] ? '1' : '0';
        $data['status']     = isset($p['status']) && $p['status'] ? '1' : '0';
        $this->Functions->update_exam((int) $id, $data);
        $this->session->setFlashdata('msg', 'Exam updated.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/exam/' . $id));
    }

    /**
     * File upload (admin AJAX). Requires admin. Returns JSON.
     */
    public function upload()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $this->response->setJSON(['ok' => false, 'error' => 'Unauthorized'])->setStatusCode(403);
        }
        $file = $this->request->getFile('file');
        if (! $file || ! $file->isValid()) {
            return $this->response->setJSON(['ok' => false, 'error' => 'No file'])->setStatusCode(400);
        }
        $path = FCPATH . 'uploads';
        if (! is_dir($path)) {
            mkdir($path, 0755, true);
        }
        $newName = $file->getRandomName();
        $file->move($path, $newName);
        return $this->response->setJSON(['ok' => true, 'filename' => $newName, 'path' => 'uploads/' . $newName]);
    }

    /**
     * File delete (admin AJAX). Requires admin. POST: file path/name.
     */
    public function delete()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $this->response->setJSON(['ok' => false])->setStatusCode(403);
        }
        $path = $this->request->getPost('file') ?? $this->request->getPost('path');
        if (! $path || strpos($path, '..') !== false) {
            return $this->response->setJSON(['ok' => false])->setStatusCode(400);
        }
        $full = FCPATH . $path;
        if (is_file($full)) {
            unlink($full);
        }
        return $this->response->setJSON(['ok' => true]);
    }

    /**
     * Send mail (admin mail form). Requires admin.
     */
    public function maildata()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/sendmail'));
        }
        $to = trim((string) $this->request->getPost('to'));
        $subj = trim((string) $this->request->getPost('subject'));
        $body = (string) $this->request->getPost('message');
        if ($to !== '' && $subj !== '') {
            $email = \Config\Services::email();
            $email->setFrom($this->Functions->get_db_option('mail_from') ?: 'noreply@certsleads.com', 'CertsLeads');
            $email->setTo($to);
            $email->setSubject($subj);
            $email->setMessage($body);
            $email->send();
        }
        $this->session->setFlashdata('msg', 'Mail sent.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/sendmail'));
    }

    /**
     * Edit date (admin date_edit form). Requires admin.
     */
    public function edit_date()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/dateedit'));
        }
        $this->Functions->update_date_settings($this->request->getPost());
        $this->session->setFlashdata('msg', 'Date settings updated.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/dateedit'));
    }

    /**
     * Update site settings. Requires admin.
     */
    public function update_settings()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/settings'));
        }
        $post = $this->request->getPost();
        foreach ($post as $name => $value) {
            if ($name !== '' && is_string($name)) {
                $this->Functions->set_db_option($name, $value);
            }
        }
        $this->session->setFlashdata('msg', 'Settings updated.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/settings'));
    }

    /**
     * Update payment gateways. Requires admin.
     */
    public function update_gateways()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/p_gateways'));
        }
        $this->Functions->update_gateways($this->request->getPost());
        $this->session->setFlashdata('msg', 'Gateways updated.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/p_gateways'));
    }

    /**
     * Update menus. Requires admin.
     */
    public function update_menus()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/menus'));
        }
        $this->Functions->update_menus($this->request->getPost());
        $this->session->setFlashdata('msg', 'Menus updated.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/menus'));
    }

    /**
     * Update sections. Requires admin.
     */
    public function update_sections()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/sections'));
        }
        $this->Functions->update_sections($this->request->getPost());
        $this->session->setFlashdata('msg', 'Sections updated.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/sections'));
    }

    /**
     * Create coupon. Requires admin.
     */
    public function create_coupon()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/coupons'));
        }
        $code = strtoupper(trim((string) $this->request->getPost('code')));
        $discount = (float) $this->request->getPost('discount');
        if ($code !== '' && $discount > 0) {
            $this->Functions->insert_coupon($code, $discount);
        }
        $this->session->setFlashdata('msg', 'Coupon created.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/coupons'));
    }

    /**
     * Create user (admin). Requires admin.
     */
    public function create_user()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/users'));
        }
        $email = trim((string) $this->request->getPost('email'));
        $name  = trim((string) $this->request->getPost('name'));
        $password = $this->request->getPost('password');
        $isAdmin = $this->request->getPost('is_admin') ? '1' : '0';
        if ($email !== '' && filter_var($email, FILTER_VALIDATE_EMAIL) && $password !== '') {
            if ($this->Functions->user('email', $email)) {
                $this->session->setFlashdata('msg', 'Email already exists.');
                $this->session->setFlashdata('msgColor', 'red');
            } else {
                $this->Functions->register($email, $name, $password);
                $u = $this->Functions->user('email', $email);
                if ($u && $isAdmin === '1') {
                    $this->Functions->set_user_admin($u->id, 1);
                }
                $this->session->setFlashdata('msg', 'User created.');
                $this->session->setFlashdata('msgColor', 'green');
            }
        }
        return redirect()->to(base_url('admin/users'));
    }

    /**
     * Edit page. Requires admin.
     */
    public function edit_page($id)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/page/' . $id));
        }
        $p = $this->request->getPost();
        $this->Functions->update_page((int) $id, [
            'title'   => $p['title'] ?? '',
            'content' => $p['content'] ?? '',
            'slug'    => $p['slug'] ?? '',
        ]);
        $this->session->setFlashdata('msg', 'Page updated.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/page/' . $id));
    }

    /**
     * Create category. POST from admin new_category form. Requires admin.
     */
    public function create_category()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/categories'));
        }
        $p = $this->request->getPost();
        $name = trim((string) ($p['name'] ?? ''));
        $slug = trim((string) ($p['slug'] ?? ''));
        if ($name !== '') {
            $this->Functions->insert_category([
                'name'         => $name,
                'slug'         => $slug ?: $name,
                'is_parent'    => isset($p['is_parent']) && $p['is_parent'] ? '1' : '0',
                'its_parent'   => (int) ($p['its_parent'] ?? 0),
                'hot_vendor'   => isset($p['hot_vendor']) && $p['hot_vendor'] ? '1' : '0',
                'top_vendor'   => isset($p['top_vendor']) && $p['top_vendor'] ? '1' : '0',
                'status'       => isset($p['status']) && $p['status'] ? '1' : '0',
                'mega_menupart' => (int) ($p['mega_menupart'] ?? 1),
            ]);
            $this->session->setFlashdata('msg', 'Category created.');
            $this->session->setFlashdata('msgColor', 'green');
        }
        return redirect()->to(base_url('admin/categories'));
    }

    /**
     * Edit category. POST from admin edit_category form. Requires admin.
     */
    public function edit_category($id)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/category/' . $id));
        }
        $p = $this->request->getPost();
        $this->Functions->update_category((int) $id, [
            'name'          => trim((string) ($p['name'] ?? '')),
            'slug'          => trim((string) ($p['slug'] ?? '')),
            'is_parent'     => isset($p['is_parent']) && $p['is_parent'] ? '1' : '0',
            'its_parent'    => (int) ($p['its_parent'] ?? 0),
            'hot_vendor'    => isset($p['hot_vendor']) && $p['hot_vendor'] ? '1' : '0',
            'top_vendor'    => isset($p['top_vendor']) && $p['top_vendor'] ? '1' : '0',
            'status'        => isset($p['status']) && $p['status'] ? '1' : '0',
            'mega_menupart' => (int) ($p['mega_menupart'] ?? 1),
        ]);
        $this->session->setFlashdata('msg', 'Category updated.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/category/' . $id));
    }

    /**
     * Delete category. GET action/delete_category/{csrf}/{id}. Requires admin.
     */
    public function delete_category($csrf, $id)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        $this->Functions->delete_category((int) $id);
        $this->session->setFlashdata('msg', 'Category deleted.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/categories'));
    }

    /**
     * Create exam. POST from admin new_exam form. Requires admin.
     */
    public function create_exam()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/exams'));
        }
        $p = $this->request->getPost();
        $title = trim((string) ($p['title'] ?? ''));
        if ($title !== '') {
            $this->Functions->insert_exam([
                'title'        => $title,
                'name'         => trim((string) ($p['name'] ?? '')),
                'slug'         => trim((string) ($p['slug'] ?? '')),
                'sku'          => $this->Functions->generate_sku(),
                'category'     => (int) ($p['category'] ?? 0),
                'sale_price'   => (float) ($p['sale_price'] ?? 0),
                'regular_price' => (float) ($p['regular_price'] ?? 0),
                'on_homepage'  => isset($p['on_homepage']) && $p['on_homepage'] ? '1' : '0',
                'hot_exam'     => isset($p['hot_exam']) && $p['hot_exam'] ? '1' : '0',
                'top_exam'     => isset($p['top_exam']) && $p['top_exam'] ? '1' : '0',
                'status'       => isset($p['status']) && $p['status'] ? '1' : '0',
            ]);
            $this->session->setFlashdata('msg', 'Exam created.');
            $this->session->setFlashdata('msgColor', 'green');
        }
        return redirect()->to(base_url('admin/exams'));
    }

    /**
     * Delete exam. GET action/delete_exam/{csrf}/{id}. Requires admin.
     */
    public function delete_exam($csrf, $id)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        $this->Functions->delete_exam((int) $id);
        $this->session->setFlashdata('msg', 'Exam deleted.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/exams'));
    }

    /**
     * Create page. POST from admin new_page form. Requires admin.
     */
    public function create_page()
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/pages'));
        }
        $p = $this->request->getPost();
        $title = trim((string) ($p['title'] ?? ''));
        if ($title !== '') {
            $this->Functions->insert_page([
                'title'   => $title,
                'content' => (string) ($p['content'] ?? ''),
                'slug'    => trim((string) ($p['slug'] ?? '')),
                'status'  => isset($p['status']) && $p['status'] ? '1' : '0',
            ]);
            $this->session->setFlashdata('msg', 'Page created.');
            $this->session->setFlashdata('msgColor', 'green');
        }
        return redirect()->to(base_url('admin/pages'));
    }

    /**
     * Edit coupon. POST from admin edit_coupon form. Requires admin.
     */
    public function edit_coupon($id)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/coupons'));
        }
        $p = $this->request->getPost();
        $this->Functions->update_coupon((int) $id, [
            'code'     => strtoupper(trim((string) ($p['code'] ?? ''))),
            'discount' => (float) ($p['discount'] ?? 0),
        ]);
        $this->session->setFlashdata('msg', 'Coupon updated.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/coupons'));
    }

    /**
     * Edit user (admin). POST from admin edit_user form. Requires admin.
     */
    public function edit_user($id)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        if ($this->request->getMethod() !== 'POST') {
            return redirect()->to(base_url('admin/users'));
        }
        $p = $this->request->getPost();
        $name = trim((string) ($p['name'] ?? ''));
        $email = trim((string) ($p['email'] ?? ''));
        $isAdmin = isset($p['is_admin']) && $p['is_admin'] ? '1' : '0';
        if ($email !== '' && filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->Functions->update_user((int) $id, [
                'name'     => $name,
                'email'    => $email,
                'is_admin' => $isAdmin,
            ]);
            if (! empty($p['password'])) {
                $this->Functions->changepass($p['password'], (int) $id);
            }
            $this->session->setFlashdata('msg', 'User updated.');
            $this->session->setFlashdata('msgColor', 'green');
        }
        return redirect()->to(base_url('admin/users'));
    }

    /**
     * Delete user. GET action/delete_user/{csrf}/{id}. Requires admin.
     */
    public function delete_user($csrf, $id)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        $this->Functions->delete_user((int) $id);
        $this->session->setFlashdata('msg', 'User deleted.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/users'));
    }

    /**
     * Delete testimonial. GET action/delete_testimonial/{csrf}/{id}. Requires admin.
     */
    public function delete_testimonial($csrf, $id)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        $this->Functions->delete_testimonial((int) $id);
        $this->session->setFlashdata('msg', 'Testimonial deleted.');
        $this->session->setFlashdata('msgColor', 'green');
        return redirect()->to(base_url('admin/testimonials'));
    }

    /**
     * Edit testimonial link. GET action/edit_testimonial/{csrf}/{id}. Redirects to admin edit. Requires admin.
     */
    public function edit_testimonial($csrf, $id)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }
        return redirect()->to(base_url('admin/testimonial/' . $id));
    }
}
