<?php

namespace App\Controllers;

/**
 * Admin controller – dashboard and admin CRUD.
 * Requires logged-in user with is_admin == 1 (see admin_login).
 */
class Admin extends BaseController
{
    /**
     * Ensure the current user is an admin; redirect to admin login if not.
     */
    protected function requireAdmin()
    {
        $userId = $this->session->get('user_id');
        if (! $userId) {
            return redirect()->to(base_url('admin_login'));
        }
        $user = $this->Functions->user('id', $userId);
        if (! $user || (string) $user->is_admin !== '1') {
            return redirect()->to(base_url('admin_login'));
        }
        return null;
    }

    /**
     * Common renderer for admin views.
     */
    protected function renderAdmin(string $view, string $title, array $extra = [])
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }

        $router = \Config\Services::router();
        $cn = $router->controllerName();
        $controller = $cn !== '' ? substr($cn, strrpos($cn, '\\') + 1) : '';
        $method = $router->methodName();

        $data = [
            'title'             => $title,
            'Functions'         => $this->Functions,
            'session'           => $this->session,
            'router_controller' => $controller,
            'router_method'     => $method,
        ] + $extra;

        return view('admin/' . $view, $data);
    }

    /**
     * Admin dashboard (stats, etc.).
     */
    public function index()
    {
        return $this->renderAdmin('index', 'Dashboard');
    }

    // ===== Exams & Categories =====

    /**
     * Exam create/edit router.
     * /admin/exam/new      -> new exam form
     * /admin/exam/{id}     -> edit exam {id}
     */
    public function exam($id = null)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }

        // prefer route param ($id); fallback to URI segment3 for robustness
        $uri      = $this->request->getUri();
        $segment3 = $id ?? $uri->getSegment(3); // 'new' or exam id

        if ($segment3 === 'new' || $segment3 === null || $segment3 === '') {
            // New exam form
            return $this->renderAdmin('new_exam', 'New Exam');
        }

        // Treat anything else as an exam ID for edit view.
        $exam = $this->Functions->exam('id', $segment3);
        if (! $exam) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound("Exam not found");
        }

        return $this->renderAdmin('edit_exam', 'Edit Exam', ['exam' => $exam]);
    }

    public function exams()
    {
        return $this->renderAdmin('exams', 'Exams');
    }

    /**
     * Category create/edit router.
     * /admin/category/new      -> new category form
     * /admin/category/{id}     -> edit category {id}
     */
    public function category($id = null)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }

        $uri      = $this->request->getUri();
        $segment3 = $id ?? $uri->getSegment(3);

        if ($segment3 === 'new' || $segment3 === null || $segment3 === '') {
            return $this->renderAdmin('new_category', 'New Category');
        }

        $category = $this->Functions->category_by_id($segment3);
        if (! $category) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound('Category not found');
        }

        return $this->renderAdmin('edit_category', 'Edit Category', ['category' => $category]);
    }

    public function categories()
    {
        return $this->renderAdmin('categories', 'Categories');
    }

    // New/edit forms (list URLs you provided)

    public function exam_new()
    {
        return $this->renderAdmin('new_exam', 'New Exam');
    }

    public function category_new()
    {
        return $this->renderAdmin('new_category', 'New Category');
    }

    public function pages()
    {
        return $this->renderAdmin('pages', 'Pages');
    }

    /**
     * Page create/edit router.
     * /admin/page/new      -> new page form
     * /admin/page/{id}     -> edit page {id}
     */
    public function page($id = null)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }

        $uri      = $this->request->getUri();
        $segment3 = $id ?? $uri->getSegment(3);

        if ($segment3 === 'new' || $segment3 === null || $segment3 === '') {
            return $this->renderAdmin('new_page', 'New Page');
        }

        $page = $this->Functions->page('id', $segment3);
        if (! $page) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound('Page not found');
        }

        return $this->renderAdmin('edit_page', 'Edit Page', ['page' => $page]);
    }

    // ===== Users & Orders =====

    public function demoonlyusers()
    {
        return $this->renderAdmin('demoonlyusers', 'Demo Only Users');
    }

    public function users()
    {
        return $this->renderAdmin('users', 'Users');
    }

    /**
     * User create/edit router.
     * /admin/user/new      -> new user form
     * /admin/user/{id}     -> edit user {id}
     */
    public function user($id = null)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }

        $uri      = $this->request->getUri();
        $segment3 = $id ?? $uri->getSegment(3);

        if ($segment3 === 'new' || $segment3 === null || $segment3 === '') {
            return $this->renderAdmin('new_user', 'New User');
        }

        $user = $this->Functions->user('id', $segment3);
        if (! $user) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound('User not found');
        }

        return $this->renderAdmin('edit_user', 'Edit User', ['user' => $user]);
    }

    public function orders()
    {
        return $this->renderAdmin('orders', 'Orders');
    }

    // ===== Coupons =====

    public function coupons()
    {
        return $this->renderAdmin('coupons', 'Coupons');
    }

    /**
     * Coupon create/edit router.
     * /admin/coupon/new      -> new coupon form
     * /admin/coupon/{id}     -> edit coupon {id}
     */
    public function coupon($id = null)
    {
        $redirect = $this->requireAdmin();
        if ($redirect !== null) {
            return $redirect;
        }

        $uri      = $this->request->getUri();
        $segment3 = $id ?? $uri->getSegment(3);

        if ($segment3 === 'new' || $segment3 === null || $segment3 === '') {
            return $this->renderAdmin('new_coupon', 'New Coupon');
        }

        $coupon = $this->Functions->coupon('id', $segment3);
        if (! $coupon) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound('Coupon not found');
        }

        return $this->renderAdmin('edit_coupon', 'Edit Coupon', ['coupon' => $coupon]);
    }

    // ===== Other admin sections =====

    public function sections()
    {
        return $this->renderAdmin('sections', 'Sections');
    }

    public function menus()
    {
        return $this->renderAdmin('menus', 'Menus');
    }

    public function media()
    {
        return $this->renderAdmin('media', 'Media');
    }

    public function p_gateways()
    {
        return $this->renderAdmin('gateways', 'Payment Gateways');
    }

    public function settings()
    {
        return $this->renderAdmin('settings', 'Settings');
    }

    public function dateedit()
    {
        return $this->renderAdmin('date_edit', 'Edit Dates');
    }

    public function sendmail()
    {
        return $this->renderAdmin('mail_form', 'Send Mail');
    }

    public function testimonials()
    {
        return $this->renderAdmin('testimonials', 'Testimonials');
    }
}
