<?php

namespace App\Controllers;

/**
 * Performance Testing Controller
 * 
 * Use this to verify that caching and optimizations are working.
 * Access at: http://localhost/certsleads-new/test-performance
 * 
 * IMPORTANT: Remove this file in production!
 */
class TestPerformance extends BaseController
{
    public function index()
    {
        $results = [];
        
        // Test 1: Settings Cache Performance
        $results['settings_cache'] = $this->testSettingsCache();
        
        // Test 2: Query Cache Performance
        $results['query_cache'] = $this->testQueryCache();
        
        // Test 3: Database Connection Type
        $results['db_connection'] = $this->testDatabaseConnection();
        
        // Test 4: Rate Limiting Status
        $results['rate_limit'] = $this->testRateLimit();
        
        // Display results
        return $this->displayResults($results);
    }
    
    /**
     * Test settings cache performance
     */
    private function testSettingsCache()
    {
        // First call - loads from database
        $start = microtime(true);
        $value1 = $this->Functions->get_db_option('site_name');
        $time1 = (microtime(true) - $start) * 1000; // Convert to milliseconds
        
        // Second call - loads from cache
        $start = microtime(true);
        $value2 = $this->Functions->get_db_option('site_name');
        $time2 = (microtime(true) - $start) * 1000;
        
        // Third call - also from cache
        $start = microtime(true);
        $value3 = $this->Functions->get_db_option('site_name');
        $time3 = (microtime(true) - $start) * 1000;
        
        $speedup = $time1 > 0 ? round($time1 / $time2, 2) : 0;
        
        return [
            'status' => $speedup > 2 ? 'PASS' : 'WARNING',
            'first_call' => round($time1, 4) . 'ms',
            'second_call' => round($time2, 4) . 'ms',
            'third_call' => round($time3, 4) . 'ms',
            'speedup' => $speedup . 'x faster',
            'message' => $speedup > 2 
                ? 'Settings cache is working perfectly!' 
                : 'Cache may not be working. Check cache configuration.'
        ];
    }
    
    /**
     * Test query cache performance
     */
    private function testQueryCache()
    {
        // Test hotvendor cache
        $start = microtime(true);
        $vendors1 = $this->Functions->hotvendor_cached(1);
        $time1 = (microtime(true) - $start) * 1000;
        
        $start = microtime(true);
        $vendors2 = $this->Functions->hotvendor_cached(1);
        $time2 = (microtime(true) - $start) * 1000;
        
        $speedup = $time1 > 0 ? round($time1 / $time2, 2) : 0;
        
        return [
            'status' => $speedup > 2 ? 'PASS' : 'WARNING',
            'first_call' => round($time1, 4) . 'ms',
            'second_call' => round($time2, 4) . 'ms',
            'speedup' => $speedup . 'x faster',
            'vendors_count' => count($vendors1),
            'message' => $speedup > 2 
                ? 'Query cache is working perfectly!' 
                : 'Query cache may not be working properly.'
        ];
    }
    
    /**
     * Test database connection type
     */
    private function testDatabaseConnection()
    {
        $db = \Config\Database::connect();
        $config = config('Database');
        
        return [
            'status' => $config->default['pConnect'] ? 'PASS' : 'FAIL',
            'persistent' => $config->default['pConnect'] ? 'YES' : 'NO',
            'driver' => $config->default['DBDriver'],
            'message' => $config->default['pConnect'] 
                ? 'Persistent connections are enabled!' 
                : 'WARNING: Persistent connections are disabled. Enable in Database.php'
        ];
    }
    
    /**
     * Test rate limiting
     */
    private function testRateLimit()
    {
        $cache = \Config\Services::cache();
        $ip = $this->request->getIPAddress();
        $key = 'rate_limit_' . md5($ip);
        
        $requests = $cache->get($key) ?? 0;
        $limit = 60; // From RateLimitFilter
        
        return [
            'status' => 'INFO',
            'current_requests' => $requests,
            'limit' => $limit,
            'remaining' => max(0, $limit - $requests),
            'message' => $requests < $limit 
                ? "Rate limiting is active. {$requests}/{$limit} requests used." 
                : 'Rate limit reached! You would be blocked on next request.'
        ];
    }
    
    /**
     * Display test results
     */
    private function displayResults($results)
    {
        $html = '<!DOCTYPE html>
<html>
<head>
    <title>Performance Test Results</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            max-width: 1200px;
            margin: 40px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        h1 {
            color: #333;
            border-bottom: 3px solid #0d9488;
            padding-bottom: 10px;
        }
        .test-section {
            background: white;
            padding: 20px;
            margin: 20px 0;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .test-section h2 {
            margin-top: 0;
            color: #0d9488;
        }
        .status {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 4px;
            font-weight: bold;
            font-size: 14px;
        }
        .status.PASS {
            background: #10b981;
            color: white;
        }
        .status.FAIL {
            background: #ef4444;
            color: white;
        }
        .status.WARNING {
            background: #f59e0b;
            color: white;
        }
        .status.INFO {
            background: #3b82f6;
            color: white;
        }
        .metric {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid #e5e7eb;
        }
        .metric:last-child {
            border-bottom: none;
        }
        .metric-label {
            font-weight: 600;
            color: #6b7280;
        }
        .metric-value {
            color: #111827;
            font-family: monospace;
        }
        .message {
            margin-top: 15px;
            padding: 12px;
            background: #f0fdfa;
            border-left: 4px solid #0d9488;
            border-radius: 4px;
        }
        .warning-box {
            background: #fef3c7;
            border-left-color: #f59e0b;
        }
        .error-box {
            background: #fee2e2;
            border-left-color: #ef4444;
        }
        .footer {
            text-align: center;
            margin-top: 40px;
            padding-top: 20px;
            border-top: 1px solid #e5e7eb;
            color: #6b7280;
        }
        .delete-warning {
            background: #fee2e2;
            border: 2px solid #ef4444;
            padding: 15px;
            border-radius: 8px;
            margin-top: 20px;
            color: #991b1b;
            font-weight: bold;
        }
    </style>
</head>
<body>
    <h1>🚀 Performance Test Results</h1>
    
    <div class="test-section">
        <h2>1. Settings Cache Performance</h2>
        <span class="status ' . $results['settings_cache']['status'] . '">' . $results['settings_cache']['status'] . '</span>
        <div class="metric">
            <span class="metric-label">First Call (Database):</span>
            <span class="metric-value">' . $results['settings_cache']['first_call'] . '</span>
        </div>
        <div class="metric">
            <span class="metric-label">Second Call (Cache):</span>
            <span class="metric-value">' . $results['settings_cache']['second_call'] . '</span>
        </div>
        <div class="metric">
            <span class="metric-label">Third Call (Cache):</span>
            <span class="metric-value">' . $results['settings_cache']['third_call'] . '</span>
        </div>
        <div class="metric">
            <span class="metric-label">Performance Improvement:</span>
            <span class="metric-value">' . $results['settings_cache']['speedup'] . '</span>
        </div>
        <div class="message">' . $results['settings_cache']['message'] . '</div>
    </div>
    
    <div class="test-section">
        <h2>2. Query Cache Performance</h2>
        <span class="status ' . $results['query_cache']['status'] . '">' . $results['query_cache']['status'] . '</span>
        <div class="metric">
            <span class="metric-label">First Call (Database):</span>
            <span class="metric-value">' . $results['query_cache']['first_call'] . '</span>
        </div>
        <div class="metric">
            <span class="metric-label">Second Call (Cache):</span>
            <span class="metric-value">' . $results['query_cache']['second_call'] . '</span>
        </div>
        <div class="metric">
            <span class="metric-label">Performance Improvement:</span>
            <span class="metric-value">' . $results['query_cache']['speedup'] . '</span>
        </div>
        <div class="metric">
            <span class="metric-label">Vendors Found:</span>
            <span class="metric-value">' . $results['query_cache']['vendors_count'] . '</span>
        </div>
        <div class="message">' . $results['query_cache']['message'] . '</div>
    </div>
    
    <div class="test-section">
        <h2>3. Database Connection</h2>
        <span class="status ' . $results['db_connection']['status'] . '">' . $results['db_connection']['status'] . '</span>
        <div class="metric">
            <span class="metric-label">Persistent Connections:</span>
            <span class="metric-value">' . $results['db_connection']['persistent'] . '</span>
        </div>
        <div class="metric">
            <span class="metric-label">Database Driver:</span>
            <span class="metric-value">' . $results['db_connection']['driver'] . '</span>
        </div>
        <div class="message ' . ($results['db_connection']['status'] === 'FAIL' ? 'error-box' : '') . '">' . $results['db_connection']['message'] . '</div>
    </div>
    
    <div class="test-section">
        <h2>4. Rate Limiting</h2>
        <span class="status ' . $results['rate_limit']['status'] . '">' . $results['rate_limit']['status'] . '</span>
        <div class="metric">
            <span class="metric-label">Current Requests:</span>
            <span class="metric-value">' . $results['rate_limit']['current_requests'] . '</span>
        </div>
        <div class="metric">
            <span class="metric-label">Rate Limit:</span>
            <span class="metric-value">' . $results['rate_limit']['limit'] . ' per minute</span>
        </div>
        <div class="metric">
            <span class="metric-label">Remaining:</span>
            <span class="metric-value">' . $results['rate_limit']['remaining'] . '</span>
        </div>
        <div class="message">' . $results['rate_limit']['message'] . '</div>
    </div>
    
    <div class="delete-warning">
        ⚠️ SECURITY WARNING: Delete this file (TestPerformance.php) in production!
    </div>
    
    <div class="footer">
        <p>Test completed at ' . date('Y-m-d H:i:s') . '</p>
        <p><a href="' . base_url() . '">← Back to Home</a></p>
    </div>
</body>
</html>';
        
        return $this->response->setBody($html);
    }
}
