<?php

namespace App\Filters;

use CodeIgniter\Filters\FilterInterface;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;

/**
 * Rate Limiting Filter
 * 
 * Prevents abuse by limiting requests per IP address.
 * Default: 60 requests per minute per IP
 */
class RateLimitFilter implements FilterInterface
{
    /**
     * Check rate limit before processing request
     */
    public function before(RequestInterface $request, $arguments = null)
    {
        $cache = \Config\Services::cache();
        $ip = $request->getIPAddress();
        $key = 'rate_limit_' . md5($ip);
        
        $requests = $cache->get($key) ?? 0;
        
        // Allow 60 requests per minute (adjust as needed)
        if ($requests > 60) {
            log_message('warning', "Rate limit exceeded for IP: {$ip}");
            
            return \Config\Services::response()
                ->setStatusCode(429)
                ->setBody('Too many requests. Please try again later.')
                ->setHeader('Retry-After', '60');
        }
        
        // Increment request counter
        $cache->save($key, $requests + 1, 60); // 60 seconds TTL
    }

    /**
     * No action needed after request
     */
    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        // Do nothing
    }
}
