<?php

namespace App\Libraries;

/**
 * Settings Cache Library
 * 
 * Caches database settings to reduce queries and improve performance.
 * All settings are loaded once and cached for 1 hour.
 */
class SettingsCache
{
    protected $cache;
    protected $db;
    protected $cacheKey = 'app_settings_cache';
    protected $cacheTTL = 3600; // 1 hour
    
    public function __construct()
    {
        $this->cache = \Config\Services::cache();
        $this->db = \Config\Database::connect();
    }
    
    /**
     * Get a setting value with caching
     * 
     * @param string $name Setting name
     * @param mixed $default Default value if not found
     * @return mixed Setting value or default
     */
    public function get($name, $default = null)
    {
        $settings = $this->getAllSettings();
        return $settings[$name] ?? $default;
    }
    
    /**
     * Get all settings from cache or database
     * 
     * @return array Associative array of settings
     */
    protected function getAllSettings()
    {
        $settings = $this->cache->get($this->cacheKey);
        
        if ($settings === null) {
            $settings = $this->loadFromDatabase();
            $this->cache->save($this->cacheKey, $settings, $this->cacheTTL);
        }
        
        return $settings;
    }
    
    /**
     * Load all settings from database
     * 
     * @return array Associative array of settings
     */
    protected function loadFromDatabase()
    {
        $builder = $this->db->table('settings');
        $query = $builder->get();
        
        $settings = [];
        if ($query) {
            foreach ($query->getResult() as $row) {
                $settings[$row->name] = $row->value;
            }
        }
        
        return $settings;
    }
    
    /**
     * Set a setting and clear cache
     * 
     * @param string $name Setting name
     * @param mixed $value Setting value
     * @return bool Success status
     */
    public function set($name, $value)
    {
        $builder = $this->db->table('settings');
        $existing = $builder->where('name', $name)->get()->getRow();
        
        if ($existing) {
            $result = $builder->where('name', $name)->update(['value' => (string) $value]);
        } else {
            $result = $builder->insert(['name' => $name, 'value' => (string) $value]);
        }
        
        // Clear cache so next request gets fresh data
        $this->clearCache();
        
        return $result;
    }
    
    /**
     * Clear settings cache
     * Call this after updating any settings in the database
     * 
     * @return bool Success status
     */
    public function clearCache()
    {
        return $this->cache->delete($this->cacheKey);
    }
    
    /**
     * Refresh cache by clearing and reloading
     * 
     * @return array Fresh settings
     */
    public function refresh()
    {
        $this->clearCache();
        return $this->getAllSettings();
    }
}
