<?php

namespace App\Models;

use CodeIgniter\Model;

class FunctionsModel extends Model
{
    protected $db;
    protected $settingsCache;
    
    public function __construct()
    {
        parent::__construct();
        $this->db = \Config\Database::connect();
        $this->settingsCache = new \App\Libraries\SettingsCache();
    }

    // ==================== USER METHODS ====================
    
    public function update_user_payment_id($id, $pid)
    {
        $builder = $this->db->table('users');
        $builder->where('id', $id);
        $builder->set('pid', $pid);
        $builder->update();
    }

    public function user($field, $value)
    {
        $builder = $this->db->table('users');
        $q = $builder->where($field, $value)->get();
        if (!$q) return null;
        return $q->getRow();
    }

    public function check_user($email, $pass)
    {
        $builder = $this->db->table('users');
        $q = $builder->where('email', $email)->get();
        $row = $q ? $q->getRow() : null;
        if (! $row) {
            return false;
        }
        return $this->validate_pw($pass, $row->password);
    }

    public function check_user_email($email)
    {
        $builder = $this->db->table('users');
        $q = $builder->where('email', $email)->get();
        if (!$q) return false;
        $row = $q->getRow();
        if (!$row || !isset($row->email)) return false;
        return $row;
    }

    public function check_user_token($token)
    {
        $builder = $this->db->table('users');
        $q = $builder->where('token', $token)->get();

        if (!$q) return false;
        $row = $q->getRow();
        if (!$row || !isset($row->token)) return false;
        return $row;
    }

    public function users($count = false)
    {
        $builder = $this->db->table('users');
        $q = $builder->orderBy('id', 'desc')->get();
        if (!$q) return $count ? 0 : [];
        
        if ($count == false) {
            return $q->getResult();
        } else {
            return count($q->getResultArray());
        }
    }

    public function register($e, $n, $p)
    {
        $builder = $this->db->table('users');
        $builder->set('email', $e);
        $builder->set('name', $n);
        $builder->set('password', $this->generate_hash($p));
        $builder->insert();
    }

    public function userTokenUpdate($email, $id)
    {
        $rand = (string) (random_int(0, PHP_INT_MAX) . $id);
        $this->db->table('users')->where('id', $id)->update(['token' => $rand]);
        return $rand;
    }

    public function userResetPassUpdate($p, $id)
    {
        $this->db->table('users')
            ->where('id', $id)
            ->update([
                'password' => $this->generate_hash($p),
                'token'    => '',
            ]);
    }

    public function update_profile($user, $n, $e, $g = '', $a = '', $c = '', $s = '', $co = '', $z = '', $v = '', $p = '', $m = '')
    {
        $this->db->table('users')
            ->where('id', $user)
            ->update([
                'name'    => $n,
                'email'   => $e,
                'gender'  => $g,
                'address' => $a,
                'city'    => $c,
                'state'   => $s,
                'country' => $co,
                'zip'     => $z,
                'vat'     => $v,
                'phone'   => $p,
                'mobile'  => $m,
            ]);
    }

    public function changepass($p, $userId)
    {
        $this->db->table('users')
            ->where('id', $userId)
            ->update(['password' => $this->generate_hash($p)]);
    }

    public function country_list(): array
    {
        return [
            '' => 'Select Your Country',
            'US' => 'United States',
            'GB' => 'United Kingdom',
            'CA' => 'Canada',
            'AU' => 'Australia',
            'IN' => 'India',
            'PK' => 'Pakistan',
            'AE' => 'United Arab Emirates',
            'SA' => 'Saudi Arabia',
            'DE' => 'Germany',
            'FR' => 'France',
            'NL' => 'Netherlands',
            'SG' => 'Singapore',
            'MY' => 'Malaysia',
            'PH' => 'Philippines',
            'NG' => 'Nigeria',
            'ZA' => 'South Africa',
            'EG' => 'Egypt',
            'KE' => 'Kenya',
            'Other' => 'Other',
        ];
    }

    // ==================== CATEGORY METHODS ====================

    public function categories($level = 'all', $limit = 10, $inArray = false, $from = 0)
    {
        $builder = $this->db->table('categories');
        
        if ($level == 'all') {
            $builder->where('id >=', $from);
            if ($limit > 0) $builder->limit($limit);
        } elseif ($level == 'parent') {
            $builder->where('is_parent', '1');
            if ($limit > 0) $builder->limit($limit);
        }
        
        $q = $builder->get();
        if (!$q) return [];
        
        return $inArray ? $q->getResultArray() : $q->getResult();
    }

    public function categories_front_all($level = 'all', $limit = 10, $inArray = false, $from = 0)
    {
        $builder = $this->db->table('categories');
        $builder->where('status', '1');
        
        if ($level == 'all') {
            $builder->where('id >=', $from);
            if ($limit > 0) $builder->limit($limit);
        } elseif ($level == 'parent') {
            $builder->where('is_parent', '1');
            if ($limit > 0) $builder->limit($limit);
        }
        
        $q = $builder->get();
        if (!$q) return [];
        
        return $inArray ? $q->getResultArray() : $q->getResult();
    }

    public function categoriesfront($level = 'all', $limit = 10, $inArray = false, $from = 0)
    {
        $builder = $this->db->table('categories');
        $builder->where('status', '1');
        
        if ($level == 'all') {
            $builder->where('id >=', $from);
            if ($limit > 0) $builder->limit($limit);
        } elseif ($level == 'parent') {
            $builder->orderBy('name', 'asc');
            $builder->where('is_parent', '1');
            if ($limit > 0) $builder->limit($limit);
        }
        
        $q = $builder->get();
        if (!$q) return [];
        
        return $inArray ? $q->getResultArray() : $q->getResult();
    }

    /**
     * Get categories with caching (1 hour)
     * Use this instead of categoriesfront() for better performance
     */
    public function categoriesfront_cached($level = 'all', $limit = 10, $inArray = false, $from = 0)
    {
        $cache = \Config\Services::cache();
        $cacheKey = "categoriesfront_{$level}_{$limit}_{$inArray}_{$from}";
        
        $result = $cache->get($cacheKey);
        if ($result === null) {
            $result = $this->categoriesfront($level, $limit, $inArray, $from);
            $cache->save($cacheKey, $result, 3600); // 1 hour
        }
        
        return $result;
    }

    public function category($field, $value)
    {
        $builder = $this->db->table('categories');
        $q = $builder->where($field, $value)->where('status', '1')->get();
        if (!$q) return null;
        return $q->getRow();
    }

    /** Get category by id (no status filter). Use for admin edit. */
    public function category_by_id($id)
    {
        $builder = $this->db->table('categories');
        $q = $builder->where('id', (int) $id)->get();
        if (! $q) return null;
        return $q->getRow();
    }

    public function category_front($field, $value)
    {
        return $this->category($field, $value);
    }

    public function categoryExam($field, $value, $status)
    {
        $builder = $this->db->table('categories');
        $q = $builder->where($field, $value)->where('status', $status)->get();
        if (!$q) return null;
        return $q->getRow();
    }

    public function hotvendor($is_parent = 1)
    {
        $builder = $this->db->table('categories');
        $q = $builder->orderBy('name', 'asc')
                     ->limit(6)
                     ->where('is_parent', '1')
                     ->where('hot_vendor', '1')
                     ->where('status', '1')
                     ->get();
        if (!$q) return [];
        return $q->getResult();
    }

    /**
     * Get hot vendors with caching (1 hour)
     * Use this instead of hotvendor() for better performance
     */
    public function hotvendor_cached($is_parent = 1)
    {
        $cache = \Config\Services::cache();
        $cacheKey = "hotvendor_{$is_parent}";
        
        $result = $cache->get($cacheKey);
        if ($result === null) {
            $result = $this->hotvendor($is_parent);
            $cache->save($cacheKey, $result, 3600); // 1 hour
        }
        
        return $result;
    }

    public function topvendor()
    {
        $builder = $this->db->table('categories');
        $q = $builder->orderBy('name', 'asc')
                     ->limit(6)
                     ->where('is_parent', '1')
                     ->where('top_vendor', '1')
                     ->where('status', '1')
                     ->get();
        if (!$q) return [];
        return $q->getResult();
    }

    /**
     * Get top vendors with caching (1 hour)
     * Use this instead of topvendor() for better performance
     */
    public function topvendor_cached()
    {
        $cache = \Config\Services::cache();
        $cacheKey = "topvendor";
        
        $result = $cache->get($cacheKey);
        if ($result === null) {
            $result = $this->topvendor();
            $cache->save($cacheKey, $result, 3600); // 1 hour
        }
        
        return $result;
    }

    public function category_header($slug)
    {
        $builder = $this->db->table('categories');
        $q = $builder->where('status', '1')->where('slug', $slug)->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function parentcategory($its_parent)
    {
        $builder = $this->db->table('categories');
        $q = $builder->where('id', $its_parent)->where('status', '1')->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function subCats($part = 'all', $c, $l = 10, $inArray = false)
    {
        $builder = $this->db->table('categories');
        $builder->where('is_parent', '0')
                ->where('status', '1')
                ->where('its_parent', $c);
        
        if ($part == 'one') {
            $builder->where('megamenu_part', 1);
        } elseif ($part == 'two') {
            $builder->where('megamenu_part', 2);
        }
        
        if ($l > 0) $builder->limit($l);
        
        $q = $builder->get();
        if (!$q) return [];
        
        return $inArray ? $q->getResultArray() : $q->getResult();
    }

    public function subCats_by_parent($cat)
    {
        $builder = $this->db->table('categories');
        $q = $builder->where('is_parent', '0')
                     ->where('its_parent', $cat)
                     ->where('status', '1')
                     ->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function search_category($qa)
    {
        $builder = $this->db->table('categories');
        $q = $builder->like('name', $qa)->where('status', '1')->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function insert_category(array $data)
    {
        $allowed = ['name', 'slug', 'is_parent', 'its_parent', 'hot_vendor', 'top_vendor', 'status', 'megamenu_part', 'image'];
        $insert = [];
        foreach ($allowed as $k) {
            if (array_key_exists($k, $data)) {
                $insert[$k] = $data[$k];
            }
        }
        if (isset($data['mega_menupart']) && ! isset($insert['megamenu_part'])) {
            $insert['megamenu_part'] = $data['mega_menupart'];
        }
        if (empty($insert)) return false;
        return $this->db->table('categories')->insert($insert);
    }

    public function update_category($id, array $data)
    {
        if (empty($data)) return false;
        $allowed = ['name', 'slug', 'is_parent', 'its_parent', 'hot_vendor', 'top_vendor', 'status', 'megamenu_part', 'image'];
        $update = [];
        foreach ($allowed as $k) {
            if (array_key_exists($k, $data)) {
                $update[$k] = $data[$k];
            }
        }
        if (isset($data['mega_menupart']) && ! array_key_exists('megamenu_part', $update)) {
            $update['megamenu_part'] = $data['mega_menupart'];
        }
        if (empty($update)) return false;
        return $this->db->table('categories')->where('id', (int) $id)->update($update);
    }

    public function delete_category($id)
    {
        return $this->db->table('categories')->where('id', (int) $id)->delete();
    }

    /** Total count of all categories (for admin pagination). */
    public function categories_total_count()
    {
        return (int) $this->db->table('categories')->countAllResults();
    }

    /** Paginated categories for admin list (all statuses, ordered by id). */
    public function categories_paginated($limit, $offset = 0)
    {
        $builder = $this->db->table('categories');
        $builder->orderBy('id', 'asc');
        if ($limit > 0) {
            $builder->limit($limit, max(0, (int) $offset));
        }
        $q = $builder->get();
        if (! $q) return [];
        return $q->getResult();
    }

    // ==================== EXAM METHODS ====================

    public function exam($field, $value)
    {
        $builder = $this->db->table('exams');
        $q = $builder->where($field, $value)->get();
        if (!$q) return null;
        return $q->getRow();
    }

    public function examEditDate($field, $value)
    {
        $builder = $this->db->table('exams');
        $q = $builder->where($field, $value)->get();
        if (!$q) return [];
        return $q->getResultArray();
    }

    public function examDemolog($field, $value)
    {
        $builder = $this->db->table('exams');
        $q = $builder->where($field, $value)->get();
        if (!$q) return [];
        return $q->getResult();
    }

    /**
     * Update exam by id. $data = associative array of column => value.
     */
    public function update_exam($id, array $data)
    {
        if (empty($data)) {
            return false;
        }
        $builder = $this->db->table('exams');
        return $builder->where('id', (int) $id)->update($data);
    }

    public function insert_exam(array $data)
    {
        if (empty($data)) return false;
        return $this->db->table('exams')->insert($data);
    }

    public function delete_exam($id)
    {
        return $this->db->table('exams')->where('id', (int) $id)->delete();
    }

    public function advanced_exams($limit = 0, $order = 'id', $category = false, $offset = 0)
    {
        $order = (is_string($order) && $order !== '') ? $order : 'id';
        $builder = $this->db->table('exams');
        $builder->where('status', '1');
        
        if ($category !== false) {
            $builder->where('category', $category);
        }
        
        if ($limit > 0) {
            $builder->limit($limit, max(0, (int) $offset));
        }
        $builder->orderBy($order, 'DESC');
        
        $q = $builder->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function hotexam($limit = 0, $order = 'id', $category = false)
    {
        $order = (is_string($order) && $order !== '') ? $order : 'id';
        $builder = $this->db->table('exams');
        $builder->where('category', $category)
                ->where('hot_exam', '1')
                ->where('status', '1');
        
        if ($limit > 0) $builder->limit($limit);
        $builder->orderBy($order, 'DESC');
        
        $q = $builder->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function topexam($limit = 0, $order = 'id', $category = false)
    {
        $order = (is_string($order) && $order !== '') ? $order : 'id';
        $builder = $this->db->table('exams');
        $builder->where('category', $category)
                ->where('top_exam', '1')
                ->where('status', '1');
        
        if ($limit > 0) $builder->limit($limit);
        $builder->orderBy($order, 'DESC');
        
        $q = $builder->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function allexam($limit = 0, $order = 'id', $category = false)
    {
        $order = (is_string($order) && $order !== '') ? $order : 'id';
        $builder = $this->db->table('exams');
        $builder->where('category', $category)->where('status', '1');
        
        if ($limit > 0) $builder->limit($limit);
        $builder->orderBy($order, 'DESC');
        
        $q = $builder->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function allsitemapexam($category)
    {
        $builder = $this->db->table('exams');
        $q = $builder->where('category', $category)->where('status', '1')->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function search_exams($qa)
    {
        $builder = $this->db->table('exams');
        $q = $builder->like('title', $qa)
                     ->orLike('name', $qa)
                     ->where('status', '1')
                     ->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function subCats_by_parent_withexam($cat, $exam)
    {
        $query = "SELECT b.* FROM `categories` as a, `exams` as b 
                  WHERE a.`its_parent`=? and a.`is_parent`='0' and a.status='1' 
                  and b.category=a.id and b.slug=? and b.status='1'";
        $q = $this->db->query($query, [$cat, $exam]);
        if (!$q || $q->getNumRows() == 0) return null;
        return $q->getRow();
    }

    public function Cats_by_parent_withexam($cat, $exam)
    {
        $query = "SELECT b.* FROM `categories` as a, `exams` as b 
                  WHERE a.`id`=? and a.`is_parent`='1' and a.status='1' 
                  and b.category=a.id and b.slug=? and b.status='1'";
        $q = $this->db->query($query, [$cat, $exam]);
        if (!$q || $q->getNumRows() == 0) return null;
        return $q->getRow();
    }

    // ==================== SETTINGS / PAGES METHODS ====================

    /**
     * Get database option with caching
     * Uses SettingsCache to reduce database queries
     */
    public function get_db_option($name)
    {
        return $this->settingsCache->get($name);
    }

    /**
     * Set or update a setting by name. Creates row if not exists.
     * Clears cache after update.
     */
    public function set_db_option($name, $value)
    {
        return $this->settingsCache->set($name, $value);
    }

    public function page($field, $value)
    {
        $builder = $this->db->table('pages');
        $q = $builder->where($field, $value)->get();
        if (!$q) return null;
        return $q->getRow();
    }

    public function pages()
    {
        $builder = $this->db->table('pages');
        $q = $builder->orderBy('id', 'DESC')->get();
        if (! $q) return [];
        return $q->getResult();
    }

    /**
     * Update page by id. Only allowed columns (title, content, slug) are written; pages table has no status column.
     */
    public function update_page($id, array $data)
    {
        $allowed = ['title', 'content', 'slug'];
        $update = [];
        foreach ($allowed as $col) {
            if (array_key_exists($col, $data)) {
                $update[$col] = $data[$col];
            }
        }
        if (empty($update)) return false;
        return $this->db->table('pages')->where('id', (int) $id)->update($update);
    }

    public function insert_page(array $data)
    {
        $allowed = ['title', 'content', 'slug'];
        $insert = [];
        foreach ($allowed as $col) {
            if (array_key_exists($col, $data)) {
                $insert[$col] = $data[$col];
            }
        }
        if (empty($insert)) return false;
        return $this->db->table('pages')->insert($insert);
    }

    /**
     * Bulk update date-related settings from POST (e.g. date_format, timezone). Saves each as option.
     */
    public function update_date_settings(array $post)
    {
        foreach (['date_format', 'timezone', 'update_label'] as $key) {
            if (array_key_exists($key, $post)) {
                $this->set_db_option($key, $post[$key]);
            }
        }
    }

    /**
     * Update payment gateway options from POST (e.g. paypal_business_email, paypal_sandbox, ziina_api_token).
     */
    public function update_gateways(array $post)
    {
        foreach (['paypal_business_email', 'paypal_sandbox', 'ziina_api_token'] as $key) {
            if (array_key_exists($key, $post)) {
                $this->set_db_option($key, $post[$key]);
            }
        }
    }

    /**
     * Update menu options from POST. Expects keys like menu_1, menu_2, ... or serialized structure.
     */
    public function update_menus(array $post)
    {
        foreach ($post as $name => $value) {
            if (is_string($name) && $name !== '' && strpos($name, 'menu') === 0) {
                $this->set_db_option($name, $value);
            }
        }
    }

    /**
     * Update section options from POST. Expects keys like section_1, section_2, ...
     */
    public function update_sections(array $post)
    {
        foreach ($post as $name => $value) {
            if (is_string($name) && $name !== '' && strpos($name, 'section') === 0) {
                $this->set_db_option($name, $value);
            }
        }
    }

    // ==================== TESTIMONIALS METHODS ====================

    public function testimonials()
    {
        $builder = $this->db->table('testimonials');
        $q = $builder->where('status', '1')->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function edit_testimonial($id)
    {
        $builder = $this->db->table('testimonials');
        $q = $builder->where('id', $id)->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function delete_testimonial($id)
    {
        $builder = $this->db->table('testimonials');
        $builder->where('id', $id)->delete();
    }

    public function add_testimonial($title, $text, $name, $date, $status)
    {
        $builder = $this->db->table('testimonials');
        return $builder->insert([
            'title' => $title,
            'text' => $text,
            'name' => $name,
            'posted_on' => $date,
            'status' => $status
        ]);
    }

    public function update_testimonial($id, $title, $text, $name, $date, $status)
    {
        $builder = $this->db->table('testimonials');
        return $builder->where('id', $id)->update([
            'text' => $text,
            'name' => $name,
            'posted_on' => $date,
            'status' => $status
        ]);
    }

    // ==================== ORDER METHODS ====================

    public function orders($key = '', $val = '', $issingle = false, $orderby = 'id')
    {
        $orderby = (is_string($orderby) && $orderby !== '') ? $orderby : 'id';
        $builder = $this->db->table('orders');
        
        if ($key && $val && $key !== '' && $val !== '') {
            $builder->where($key, $val);
        }
        
        $builder->orderBy($orderby, 'DESC');
        $q = $builder->get();
        
        if (!$q) return $issingle ? null : [];
        
        return $issingle ? $q->getRow() : $q->getResultArray();
    }

    public function add_order($charged, $sku, $user, $coupon = '', $date)
    {
        $builder = $this->db->table('orders');
        return $builder->insert([
            'charged' => $charged,
            'sku' => $sku,
            'user' => $user,
            'coupon' => $coupon,
            'date' => $date
        ]);
    }

    public function coupon($key, $value)
    {
        $builder = $this->db->table('coupons');
        $q = $builder->where($key, $value)->get();
        if (!$q) return null;
        return $q->getRow();
    }

    public function coupons()
    {
        $builder = $this->db->table('coupons');
        $q = $builder->orderBy('id', 'DESC')->get();
        if (! $q) return [];
        return $q->getResult();
    }

    /**
     * Insert a new coupon. Expects code (string) and discount (numeric, e.g. percentage).
     */
    public function insert_coupon($code, $discount)
    {
        $builder = $this->db->table('coupons');
        return $builder->insert([
            'code' => $code,
            'discount' => (float) $discount,
        ]);
    }

    public function update_coupon($id, array $data)
    {
        if (empty($data)) return false;
        return $this->db->table('coupons')->where('id', (int) $id)->update($data);
    }

    /**
     * Set user admin flag. $value = '1' or 1 for admin, '0' or 0 for not.
     */
    public function set_user_admin($userId, $value)
    {
        $this->db->table('users')
            ->where('id', (int) $userId)
            ->update(['is_admin' => (string) $value === '1' ? '1' : '0']);
    }

    public function update_user($id, array $data)
    {
        if (empty($data)) return false;
        $allowed = ['name', 'email', 'is_admin'];
        $update = [];
        foreach ($allowed as $k) {
            if (array_key_exists($k, $data)) {
                $update[$k] = $data[$k];
            }
        }
        if (empty($update)) return false;
        return $this->db->table('users')->where('id', (int) $id)->update($update);
    }

    public function delete_user($id)
    {
        return $this->db->table('users')->where('id', (int) $id)->delete();
    }

    /**
     * Return pipe-separated SKUs from session cart (for checkout). Replaces CI3 Cart get_sku_cart.
     */
    public function get_sku_cart(): string
    {
        $session = \Config\Services::session();
        $cart = $session->get('cart') ?? [];
        if (!is_array($cart) || empty($cart)) {
            return '';
        }
        $skus = [];
        foreach ($cart as $item) {
            $sku = is_array($item) ? ($item['sku'] ?? $item['id'] ?? '') : ($item->sku ?? $item->id ?? '');
            if ($sku !== '') {
                $skus[] = $sku;
            }
        }
        return implode('|', $skus);
    }

    // ==================== DOWNLOADS METHODS ====================

    public function downloads($key, $value)
    {
        $builder = $this->db->table('downloads');
        $q = $builder->where($key, $value)->get();
        if (!$q) return [];
        return $q->getResult();
    }

    public function add_download($order, $product, $user, $date, $ip)
    {
        $builder = $this->db->table('downloads');
        return $builder->insert([
            'order' => $order,
            'product' => $product,
            'user' => $user,
            'date' => $date,
            'ip' => $ip
        ]);
    }

    // ==================== HELPER / UTILITY METHODS ====================

    public function save_email_id($e, $n, $exam)
    {
        $builder = $this->db->table('demo_download_log');
        $now = date("Y-m-d H:i:s");
        return $builder->insert([
            'email_id' => $e,
            'username' => $n,
            'exam_id' => $exam,
            'user_ip' => $_SERVER['REMOTE_ADDR'] ?? '',
            'log_datetime' => $now
        ]);
    }

    public function demo_download_log()
    {
        $query = "SELECT DISTINCT (email_id), exam_id, user_ip, log_datetime, username 
                  FROM demo_download_log ORDER BY log_datetime DESC";
        $q = $this->db->query($query);
        if (!$q || $q->getNumRows() == 0) return null;
        return $q->getResult();
    }

    public function recent_updated_categories($limit)
    {
        $builder = $this->db->table('exams');
        $q = $builder->orderBy('update', 'DESC')->get();
        
        $categories = [];
        if (!$q) return $categories;
        
        foreach ($q->getResult() as $a) {
            if (count($categories) > 3 || in_array($a->category, $categories)) {
                break;
            }
            $categories[] = $a->category;
        }
        return $categories;
    }

    public function get_all_exams_inc_subs($cat)
    {
        $subCats = $this->subCats('all', $cat, 0);
        $all = [];
        $q = $this->advanced_exams(0, 'id', $cat);
        $all[] = $q;
        
        foreach ($subCats as $sub) {
            $aq = $this->advanced_exams(0, 'id', $sub->id);
            if ($aq && count($aq) >= 1) {
                $all[] = $aq;
            }
        }
        return $all;
    }

    public function is_already_bought($exam)
    {
        $session = session();
        $u = $session->get('user_id');
        if (!isset($u) || !$u) return false;
        
        $orders = $this->orders('user', $u, false, 'date');
        $flag = 0;
        
        foreach ($orders as $order) {
            if ($flag !== 0) break;
            
            $skuz = explode('|', $order['sku']);
            foreach ($skuz as $sku) {
                if ($sku == '') break;
                $productDetails = $this->exam('sku', $sku);
                if (!$productDetails || $productDetails->id !== $exam) continue;
                
                if ($this->check_update_period($order['id'])) {
                    $flag = $order['id'];
                }
            }
        }
        
        return $flag == 0 ? false : $flag;
    }

    public function check_update_period($orderrr)
    {
        $session = session();
        $u = $session->get('user_id');
        if (!isset($u) || !$u) return false;
        
        $orders = $this->orders('user', $u, false, 'date');
        foreach ($orders as $order) {
            if ($order['id'] != $orderrr) continue;
            
            $diff = date_diff(date_create($order['date']), date_create(date('Y-m-d H:i:s')));
            return $diff->days <= 90;
        }
        return false;
    }

    public function generate_sku()
    {
        helper('text');
        $builder = $this->db->table('exams');
        $q = $builder->get();
        
        $skus = [];
        if ($q) {
            foreach ($q->getResult() as $skua) {
                $skus[] = $skua->sku;
            }
        }
        
        $sku = random_string('alnum', 5);
        while (in_array($sku, $skus)) {
            $sku = random_string('alnum', 5);
        }
        return $sku;
    }

    // ==================== PASSWORD METHODS ====================

    private function generate_hash($password, $cost = 11)
    {
        $salt = substr(base64_encode(openssl_random_pseudo_bytes(17)), 0, 22);
        $salt = str_replace("+", ".", $salt);
        $param = '$' . implode('$', ['2y', str_pad($cost, 2, "0", STR_PAD_LEFT), $salt]);
        return crypt($password, $param);
    }

    private function validate_pw($password, $hash)
    {
        return crypt($password, $hash) === $hash;
    }

    // ==================== DASHBOARD / SALE HELPERS ====================

    /** Count of customers (for admin dashboard). */
    public function customers_sale($what = 'count')
    {
        if ($what === 'count') {
            return $this->users(true);
        }
        return 0;
    }

    /** Aggregate sale total (for admin dashboard). */
    public function aggregrate_sale($formatted = false)
    {
        $builder = $this->db->table('orders');
        $builder->selectSum('charged');
        $q = $builder->get();
        $row = $q ? $q->getRow() : null;
        $total = $row && isset($row->charged) ? (float) $row->charged : 0;
        return $formatted ? number_format($total, 2) : $total;
    }

    // ==================== TEMPLATE METHOD ====================

    public function template($view, $data = [])
    {
        $theme = $this->get_db_option('active-theme') ?? 'default';
        return view($theme . '/' . $view, $data);
    }
}
